/*********************************************************************
*	 项目名称:MG82F6P32-DEMO
*			测试板使用 TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV Board
*			CpuCLK=12MHz, SysCLK=12MHz
*	功能描述：
*			CPUCLK=SYSCLK (12M)
*			PCA时钟: SYSCLK
*			CH0(P22),CH1(P23),CH2(P24),CH3(P34),CH4(P32),CH5(P35)
*			在PCA中断里修改占空比
*			边沿对齐,分辨率为1000
*			PWM频率: PCAClk/(1000)=12MHz/1000=12KHz
*
*			P10作为输入控制PWM输出，P10=1: 有输出， P10=0: 无输出
*	注意事项：
*			非缓冲模式,应在PCA计数器溢出中断里更改占空比.
*	 建立时间: 2024.2.26
*	 修改日志:
*
*********************************************************************/
#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*定义系统时钟SysClk (MAX.50MHz)
*可选:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*定义CPU时钟 CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/
#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	TIMER_12T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	/256)
#define	TIMER_12T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(12000000))))	%256)

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	PWM_MIN			(0*1)
#define	PWM_MAX			(1000*1)
#define	PWM_3_4			(750*1)
#define	PWM_2_4			(500*1)
#define	PWM_1_4			(250*1)
#define	PWM_LOW			(40*1)
#define	PWM_HINT		(50*1)

#define	PTM0_RELOAD		(PWM_MAX)

#define	PTM0_C			 (65536)

#define	PTM0_CL(x)		(u8)((~(u16)x)+1)
#define	PTM0_CH(x)		(u8)(((~(u16)x)+1)>>8)

idata WordTypeDef wDuty[6];
bit	bDutyChange;
u8 DutyFlag;

/***********************************************************************************
*函数名称:	 void INT_PCA(void)
*功能描述:PCA 中断服务程序

*输入参数:
*返回参数:
*************************************************************************************/
void INT_PTM0(void)	interrupt INT_VECTOR_PTM0
{
	WordTypeDef	duty;
	_push_(SFRPI);
	SFRPI=0;
	if(CF)
	{
		IO_LED_R=!IO_LED_R;
		CF=0;
		// Todo...
		// ......
		if(bDutyChange)
		{
			duty.W=PTM0_C-wDuty[0].W;
			PTM0_CH0_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PTM0_C-wDuty[1].W;
			PTM0_CH1_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PTM0_C-wDuty[2].W;
			PTM0_CH2_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PTM0_C-wDuty[3].W;
			PTM0_CH3_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PTM0_C-wDuty[4].W;
			PTM0_CH4_SetValue(duty.B.BHigh,duty.B.BLow);
			duty.W=PTM0_C-wDuty[5].W;
			PTM0_CH5_SetValue(duty.B.BHigh,duty.B.BLow);
			bDutyChange=FALSE;
		}
	}
	_pop_(SFRPI);
}

/*************************************************
*函数名称:	   void	DelayXus(u16 xUs)
*功能描述:		延时程序，单位为us
*输入参数:	   u8 Us ->	*1us  (1~255)
*返回参数:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*函数名称:	   void	DelayXms(u16 xMs)
*功能描述:	   延时程序，单位为ms
*输入参数:	   u16 xMs -> *1ms	(1~65535)
*返回参数:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*函数名称:	 void InitPort(void)
*功能描述:	 IO配置
*输入参数:
*返回参数:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT6);						// 设置P26为推挽输出(LED)
	PORT_SetP2PushPull(BIT2|BIT3|BIT4);				// 设置P22(CEX0),P23(CEX2),P24(CEX4)为推挽输出
	PORT_SetP3PushPull(BIT2|BIT4|BIT5);				// 设置P32(CEX1),P34(CEX3),P35(CEX5)为推挽输出

	P10=1;
	PORT_SetP1OpenDrainPu(BIT0);					// 设置P10为开漏带上拉，用于控制输入
}

/***********************************************************************************
*函数名称:	 void InitPTM0_PWM(void)
*功能描述:	 PCA初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitPTM0_PWM(void)
{
	PTM0_SetCLOCK_SYSCLK();					// PCA时钟为SysClk	= 12MHz

	PTM0_CH0_SetMode_PWM();
	PTM0_CH1_SetMode_PWM();
	PTM0_CH2_SetMode_PWM();
	PTM0_CH3_SetMode_PWM();
	PTM0_CH4_SetMode_PWM();
	PTM0_CH5_SetMode_PWM();

	PTM0_CH0_SetPWM_16Bit();
	PTM0_CH1_SetPWM_16Bit();
	PTM0_CH2_SetPWM_16Bit();
	PTM0_CH3_SetPWM_16Bit();
	PTM0_CH4_SetPWM_16Bit();
	PTM0_CH5_SetPWM_16Bit();

	PTM0_SetPWM_EdgeAligned();				// 边沿对齐

	// 设置PWM占空比比较值
	PTM0_CH0_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH1_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH2_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH3_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH4_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));
	PTM0_CH5_SetValue(PTM0_CH(PWM_MIN),PTM0_CL(PWM_MIN));

	// 设置PWM频率
	PTM0_SetCounter(PTM0_C-PTM0_RELOAD);
	PTM0_SetCounterReload(PTM0_C-PTM0_RELOAD);

	// 禁止PWM输出
	PTM0_SetPWM0_DisOutput();
	PTM0_SetPWM1_DisOutput();
	PTM0_SetPWM2_DisOutput();
	PTM0_SetPWM3_DisOutput();
	PTM0_SetPWM4_DisOutput();
	PTM0_SetPWM5_DisOutput();

	PTM0_SetPOEn_PWMCycle();				// 禁止/使能PWM输出功能对齐到PWM周期

	// 设置PWM输出引脚
	PTM0_SetCEXn_Pin(CEX0_P22|CEX1_P23|CEX2_P24|CEX3_P34|CEX4_P32|CEX5_P35);

	PTM0_CF_EnInterrupt();					// 使能CF中断

	PTM0_EnPCACounter();					// 使能PCA计数器,

}

/***********************************************************************************
*函数名称:	 void InitInterrupt(void)
*功能描述:	 中断初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitInterrupt(void)
{
	INT_EnPTM0();							// 使能PCA中断

}

/***********************************************************************************
*函数名称:	 void InitClock(void)
*功能描述:	 时钟初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 输出 MCK/4
	//CLK_P60OC_MCKDiv4();
}


/***********************************************************************************
*函数名称:	 void InitSystem(void)
*功能描述:	 系统初始化
*输入参数:
*返回参数:
*************************************************************************************/
void InitSystem(void)
{
	InitClock();
	InitPort();
	InitPTM0_PWM();
	InitInterrupt();

}


void main()
{
	u8 i,x,cnt;
	bit	bOutFlag;
	InitSystem();

	IO_LED_G=0;
	DelayXms(1000);
	IO_LED_G=1;

	INT_EnAll();


	wDuty[0].W=PWM_MIN;
	wDuty[1].W=PWM_LOW;
	wDuty[2].W=PWM_1_4;
	wDuty[3].W=PWM_2_4;
	wDuty[4].W=PWM_3_4;
	wDuty[5].W=PWM_2_4;

	DutyFlag=0x00;
	bOutFlag=FALSE;

	while(1)
	{
		DelayXms(1);
		cnt++;
		if(cnt>=200)
		{
			cnt=0;
			IO_LED_G=!IO_LED_G;
			if(bOutFlag==TRUE)
			{
				while(bDutyChange==TRUE);		//等待之前的数据更改完成
				x=0x01;
				for(i=0;i<8;i++)
				{
					if((DutyFlag&x)==0)
					{
						wDuty[i].W=wDuty[i].W+20;
						if(wDuty[i].W >= PWM_MAX)
						{
							wDuty[i].W = PWM_MAX;
							DutyFlag=DutyFlag|x;
						}
					}
					else
					{
						if(wDuty[i].W <	21)
						{
							wDuty[i].W = PWM_MIN;
							DutyFlag=DutyFlag&(~x);
						}
						else
						{
							wDuty[i].W=wDuty[i].W-20;
						}
					}
					x=x<<1;
				}
				bDutyChange=TRUE;
			}
		}
		if(P10==1)
		{
			if(bOutFlag==FALSE)
			{
				bOutFlag=TRUE;
				// 使能PWM输出
				PTM0_SetPWM0_EnOutput();
				PTM0_SetPWM1_EnOutput();
				PTM0_SetPWM2_EnOutput();
				PTM0_SetPWM3_EnOutput();
				PTM0_SetPWM4_EnOutput();
				PTM0_SetPWM5_EnOutput();
			}
		}
		else
		{
			if(bOutFlag==TRUE)
			{
				bOutFlag=FALSE;
				// 禁止PWM输出
				PTM0_SetPWM0_DisOutput();
				PTM0_SetPWM1_DisOutput();
				PTM0_SetPWM2_DisOutput();
				PTM0_SetPWM3_DisOutput();
				PTM0_SetPWM4_DisOutput();
				PTM0_SetPWM5_DisOutput();
			}
		}
	}
}

